<?php
/**
 * Plugin Name: WooCommerce Dynamic Dimensions Price
 * Description: Berechnet Produktpreise nach m² anhand von Breite/Höhe in cm. Zeigt live Preis & Fläche an, speichert Daten in Bestellung.
 * Version: 1.4
 * Author: Dein Name
 */

if (!defined('ABSPATH')) exit;

add_action('plugins_loaded', 'wdd_init');
function wdd_init() {
    if (!class_exists('WooCommerce')) {
        add_action('admin_notices', function() {
            echo '<div class="error"><p><strong>WooCommerce Dynamic Dimensions:</strong> WooCommerce ist nicht aktiv.</p></div>';
        });
        return;
    }

    // Produkt-Option
    add_action('woocommerce_product_options_general_product_data', 'wdd_add_checkbox');
    add_action('woocommerce_process_product_meta', 'wdd_save_checkbox');

    // Felder im Frontend
    add_action('woocommerce_before_add_to_cart_button', 'wdd_show_fields');
    add_action('wp_enqueue_scripts', 'wdd_enqueue_scripts');

    // Daten in Warenkorb übernehmen
    add_filter('woocommerce_add_cart_item_data', 'wdd_add_cart_item_data', 10, 2);
    add_action('woocommerce_before_calculate_totals', 'wdd_before_calculate_totals', 10, 1);

    // Anzeige im Warenkorb/Checkout
    add_filter('woocommerce_get_item_data', 'wdd_show_cart_item_data', 10, 2);

    // Anzeige in Bestell-E-Mails & Admin
    add_action('woocommerce_checkout_create_order_line_item', 'wdd_add_order_item_meta', 10, 4);
}

/**
 * Checkbox im Backend
 */
function wdd_add_checkbox() {
    woocommerce_wp_checkbox(array(
        'id' => '_wdd_enabled',
        'label' => __('Preis nach m² berechnen', 'wdd'),
        'description' => __('Aktivieren, um Preis anhand von Breite x Höhe (cm) zu berechnen.', 'wdd'),
    ));
}

function wdd_save_checkbox($post_id) {
    $value = isset($_POST['_wdd_enabled']) ? 'yes' : 'no';
    update_post_meta($post_id, '_wdd_enabled', $value);
}

/**
 * Eingabefelder im Produkt
 */
function wdd_show_fields() {
    global $post;
    if (empty($post->ID)) return;
    $product = wc_get_product($post->ID);
    if (!$product) return;

    $enabled = get_post_meta($product->get_id(), '_wdd_enabled', true);
    if ($enabled !== 'yes') return;

    echo '<div class="wdd-dimensions" style="margin:10px 0;">';
    echo '<p><label for="wdd_width">Breite (cm): <input type="number" id="wdd_width" name="wdd_width" min="0" step="1" required></label></p>';
    echo '<p><label for="wdd_height">Höhe (cm): <input type="number" id="wdd_height" name="wdd_height" min="0" step="1" required></label></p>';
    echo '<p id="wdd_area" style="font-weight:bold; margin-top:10px;"></p>';
    echo '</div>';
}

/**
 * JS einbinden
 */
function wdd_enqueue_scripts() {
    if (!is_product()) return;
    global $post;
    if (empty($post->ID)) return;
    $product = wc_get_product($post->ID);
    if (!$product) return;

    $enabled = get_post_meta($product->get_id(), '_wdd_enabled', true);
    if ($enabled !== 'yes') return;

    wp_enqueue_script('wdd-script', plugin_dir_url(__FILE__) . 'wdd.js', array('jquery'), '1.4', true);

    $sqm_price = $product->get_regular_price();
    if ($sqm_price === '' || is_null($sqm_price)) {
        $sqm_price = $product->get_price();
    }
    $sqm_price = $sqm_price !== '' ? (float) $sqm_price : 0.0;

    wp_localize_script('wdd-script', 'wdd_vars', array(
        'sqm_price' => $sqm_price,
        'currency_symbol' => get_woocommerce_currency_symbol(),
        'price_per_sqm_text' => wc_price($sqm_price) . ' ' . __('pro m² inkl. MwSt. zzgl. Versand', 'wdd'),
    ));
}

/**
 * Daten in den Warenkorb
 */
function wdd_add_cart_item_data($cart_item_data, $product_id) {
    $enabled = get_post_meta($product_id, '_wdd_enabled', true);
    if ($enabled === 'yes' && isset($_POST['wdd_width'], $_POST['wdd_height'])) {
        $width = floatval($_POST['wdd_width']);
        $height = floatval($_POST['wdd_height']);
        if ($width > 0 && $height > 0) {
            $sqm = ($width / 100) * ($height / 100); // Fläche in m²

            $cart_item_data['wdd_width']  = $width;
            $cart_item_data['wdd_height'] = $height;
            $cart_item_data['wdd_sqm']    = $sqm;
            $cart_item_data['unique_key'] = md5(microtime() . rand());
        }
    }
    return $cart_item_data;
}

/**
 * Preis im Warenkorb anpassen
 */
function wdd_before_calculate_totals($cart) {
    if (is_admin() && !defined('DOING_AJAX')) return;
    if (!is_object($cart) || !method_exists($cart, 'get_cart')) return;

    foreach ($cart->get_cart() as $cart_item) {
        if (isset($cart_item['wdd_width'], $cart_item['wdd_height'], $cart_item['wdd_sqm'])) {
            $product = wc_get_product($cart_item['product_id']);
            if (!$product) continue;

            $sqm_price = $product->get_regular_price();
            if ($sqm_price === '' || is_null($sqm_price)) $sqm_price = $product->get_price();
            $sqm_price = (float) $sqm_price;

            $new_price = $sqm_price * $cart_item['wdd_sqm'];
            $cart_item['data']->set_price(max(0, $new_price));
        }
    }
}

/**
 * Anzeige im Warenkorb & Checkout
 */
function wdd_show_cart_item_data($item_data, $cart_item) {
    if (isset($cart_item['wdd_width'], $cart_item['wdd_height'], $cart_item['wdd_sqm'])) {
        $item_data[] = array(
            'name' => 'Breite (cm)',
            'value' => intval($cart_item['wdd_width']),
        );
        $item_data[] = array(
            'name' => 'Höhe (cm)',
            'value' => intval($cart_item['wdd_height']),
        );
        $item_data[] = array(
            'name' => 'Fläche (m²)',
            'value' => wc_format_decimal($cart_item['wdd_sqm'], 2),
        );
    }
    return $item_data;
}

/**
 * In Bestellung (Admin + E-Mail)
 */
function wdd_add_order_item_meta($item, $cart_item_key, $values, $order) {
    if (isset($values['wdd_width'], $values['wdd_height'], $values['wdd_sqm'])) {
        $item->add_meta_data('Breite (cm)', intval($values['wdd_width']));
        $item->add_meta_data('Höhe (cm)', intval($values['wdd_height']));
        $item->add_meta_data('Fläche (m²)', wc_format_decimal($values['wdd_sqm'], 2));
    }
}
/**
 * Preis im Produktarchiv ersetzen (nur für m²-Produkte)
 */
add_filter('woocommerce_get_price_html', 'wdd_archive_price_html', 20, 2);
function wdd_archive_price_html($price, $product) {
    $enabled = get_post_meta($product->get_id(), '_wdd_enabled', true);
    if ($enabled === 'yes') {
        $sqm_price = $product->get_regular_price();
        if ($sqm_price === '' || is_null($sqm_price)) {
            $sqm_price = $product->get_price();
        }
        $sqm_price = $sqm_price !== '' ? (float) $sqm_price : 0.0;

        if ($sqm_price > 0) {
            $price = wc_price($sqm_price) . ' ' . __('pro m² inkl. MwSt. zzgl. Versand', 'wdd');
        }
    }
    return $price;
}

/**
 * Button im Produktarchiv ändern (nur für m²-Produkte)
 */
add_filter('woocommerce_loop_add_to_cart_link', 'wdd_archive_button', 20, 2);
function wdd_archive_button($button, $product) {
    $enabled = get_post_meta($product->get_id(), '_wdd_enabled', true);
    if ($enabled === 'yes') {
        $url   = get_permalink($product->get_id());
        $label = __('Preis berechnen', 'wdd');
        $button = '<a href="' . esc_url($url) . '" class="button wdd-calc-button">' . esc_html($label) . '</a>';
    }
    return $button;
}